package KTEditor;

import java.io.Serializable;
import java.util.Hashtable;
import java.awt.Dimension;
import java.awt.Color;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/**
 * Object describing type used by a parameter to a kinetic typography effect.
 * xx more later
 * <p>
 *
 * @author  Scott Hudson
 */
public abstract class ParameterTypeDescriptor implements Serializable {
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* instance variables */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Name of this type.  This should be something suitable
     * for display to the user
     */
    protected String name = "$$UNNAMED_TYPE$$";
    
    /** Return the name of the type. */
    public String getName() {return name;}
    
    /** Set the name of the type. */
    public void setName(String nm) {name = nm;}
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* default layout constants for generated UIs */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Default/recommended width for parameter interfaces generated by subclasses. */
    public static final int DEFAULT_OUTER_W = 275;
    /** Default/recommended height for parameter interfaces generated by subclasses. */
    public static final int DEFAULT_OUTER_H = 55;
    /** Default/recommended size for parameter interfaces generated by subclasses. */
    public static final Dimension DEFAULT_OUTER_SIZE = new Dimension(DEFAULT_OUTER_W,DEFAULT_OUTER_H);
    
    /** Amount to indent initial row of controls in packed layout */
    public static final int DEFAULT_INDENT_W = 10;
    /** Amount to indent initial row of controls in packed layout */
    public static final Dimension DEFAULT_INDENT = new Dimension(DEFAULT_INDENT_W, 2);
    
    /** Default/recommended total width for parameter controls generated by subclasses. */
    public static final int DEFAULT_CONTROL_W = DEFAULT_OUTER_W-10-DEFAULT_INDENT_W;
    /** Default/recommended height for parameter controls generated by subclasses. */
    public static final int DEFAULT_CONTROL_H = 45;
    /** Default/recommended size for parameter controls generated by subclasses. */
    public static final Dimension DEFAULT_CONTROL_SIZE = new Dimension(DEFAULT_CONTROL_W,DEFAULT_CONTROL_H);
    
    /** Default/recommended width for the major parameter control sharing space with other controls. */
    public static final int DEFAULT_PRIME_CONTROL_W = DEFAULT_CONTROL_W - 75;
    /** Default/recommended height for the major parameter control sharing space with other controls. */
    public static final int DEFAULT_PRIME_CONTROL_H = DEFAULT_CONTROL_H;
    /** Default/recommended size for the major parameter control sharing space with other controls. */
    public static final Dimension DEFAULT_PRIME_CONTROL_SIZE = new Dimension(DEFAULT_PRIME_CONTROL_W,DEFAULT_PRIME_CONTROL_H);
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Create and return an interface for editing a value of this type.  The given object serves as the
     * model holding the value being edited.  The type of the object depends on the subclass implementing
     * this method but would typically be a DoubleHolder, IntHolder, or BooleanHolder.  The string
     * parameter here provides a human readable indication of what value is being edited.  Warning:
     * Once an interface has been built with this method it does not get updated if the type is changed
     * (e.g., if the min or max value of a range type is modified).
     */
    abstract public JComponent buildInterface(Object model, String lab);
    
    //xx ***************** DEAD CODE **************************
    /**
     * This method takes one or more components which are to form the interface for a parameter control
     * (e.g., being created by buildInterface()), places them in an optionally labeled JPanel, and applies
     * standard sizing.  Unneeded components can be coded as null and if the label is null it is assumed
     * that the label is already embedded in one of the components (e.g., in a checkbox).
     */
    public static JComponent xxOLDpackageInterface(  
    JComponent part1,
    JComponent part2,
    JComponent part3,
    JComponent part4,
    String lab) {
        // For uniformity of sizing we force a label
        if (lab == null) lab = " ";
        
        // Result we be a panel holding all this using a left justified flow layout
        JPanel result = new JPanel(new FlowLayout(FlowLayout.LEFT));
        
        // single part?
        if (part1 != null && part2 == null && part3 == null && part4 == null) {
            part1.setPreferredSize(DEFAULT_CONTROL_SIZE);
            part1.setMinimumSize(DEFAULT_CONTROL_SIZE);
        }
        else {   // multiple parts
            // give part1 a preferred size so it gets preference
            if (part1 != null) {
                part1.setPreferredSize(DEFAULT_PRIME_CONTROL_SIZE);
                part1.setMinimumSize(DEFAULT_PRIME_CONTROL_SIZE);
            }
        }
        
        // add all the parts we have
        if (part1 != null) result.add(part1);
        if (part2 != null) result.add(part2);
        if (part3 != null) result.add(part3);
        if (part4 != null) result.add(part4);
        
        // label it
        result.setBorder(new TitledBorder(lab));
        
        // size overall result
        result.setPreferredSize(DEFAULT_OUTER_SIZE);
        result.setMinimumSize(DEFAULT_OUTER_SIZE);
        result.setMaximumSize(DEFAULT_OUTER_SIZE);
        
        return result;
    }
    
    
    //xx ************* END DEAD CODE
    
    /**
     * This method takes one or more components which are to form the interface for a parameter control
     * (e.g., being created by buildInterface()), places them in an optionally labeled JPanel, and applies
     * standard sizing.  Unneeded components can be coded as null and if the label is null it is assumed
     * that the label is already embedded in one of the components (e.g., in a checkbox).
     */
    public static JComponent packageInterface(
    JComponent part1,
    JComponent part2,
    JComponent part3,
    JComponent part4,
    String lab) {
        // For uniformity of sizing we force a label
        if (lab == null) lab = " ";
        
        // make a label
        JComponent part0 = new JLabel(lab);
        Box labBox = Box.createHorizontalBox();
        labBox.add(Box.createHorizontalStrut(0));
        labBox.add(part0);
        labBox.add(Box.createHorizontalGlue());
        
        // Result we be a veritical box 
        // A box with the label will be the first item, followed by a flow panel with the controls
        Box result = Box.createVerticalBox();
        
        // single part?
        if (part1 != null && part2 == null && part3 == null && part4 == null) {
            part1.setPreferredSize(DEFAULT_CONTROL_SIZE);
            part1.setMinimumSize(DEFAULT_CONTROL_SIZE);
        }
        else {   // multiple parts
            // give part1 a preferred size so it gets preference
            if (part1 != null) {
                part1.setPreferredSize(DEFAULT_PRIME_CONTROL_SIZE);
                part1.setMinimumSize(DEFAULT_PRIME_CONTROL_SIZE);
            }
        }
        
        // add all the parts we have, label is first 
        result.add(Box.createVerticalStrut(0));
        result.add(labBox);
        result.add(Box.createVerticalStrut(2));
        
        // rest of controls go in their own left justified flow
        JPanel controls = new JPanel(new FlowLayout(FlowLayout.LEFT, 2, 1));
        controls.setBorder(BorderFactory.createEmptyBorder());
        controls.add(new Box.Filler(DEFAULT_INDENT, DEFAULT_INDENT, DEFAULT_INDENT));
        if (part1 != null) controls.add(part1);
        if (part2 != null) controls.add(part2);
        if (part3 != null) controls.add(part3);
        if (part4 != null) controls.add(part4);
        result.add(controls);
        result.add(Box.createVerticalGlue());
               
        // size overall result
        result.setPreferredSize(DEFAULT_OUTER_SIZE);
        result.setMinimumSize(DEFAULT_OUTER_SIZE);
        result.setMaximumSize(DEFAULT_OUTER_SIZE);
        
        return result;
    }
    
    /**
     * This method takes a component which is to form the interface for a parameter control
     * (e.g., being created by buildInterface()), places it in an optionally labeled JPanel, and applies
     * standard sizing.  If the label is null it is assumed that the label is already embedded in one
     * of the components (e.g., in a checkbox).
     */
    public static JComponent packageInterface(JComponent part1, String lab) {
        return packageInterface(part1,null,null,null,lab);
    }
    
    /**
     * This method takes one or more components which are to form the interface for a parameter control
     * (e.g., being created by buildInterface()), places them in an optionally labeled JPanel, and applies
     * standard sizing.  Unneeded components can be coded as null and if the label is null it is assumed
     * that the label is already embedded in one of the components (e.g., in a checkbox).
     */
    public static JComponent packageInterface(JComponent part1, JComponent part2, String lab) {
        return packageInterface(part1,part2,null,null,lab);
    }
    
    /**
     * This method takes one or more components which are to form the interface for a parameter control
     * (e.g., being created by buildInterface()), places them in an optionally labeled JPanel, and applies
     * standard sizing.  Unneeded components can be coded as null and if the label is null it is assumed
     * that the label is already embedded in one of the components (e.g., in a checkbox).
     */
    public static JComponent packageInterface(JComponent part1, JComponent part2, JComponent part3, String lab) {
        return packageInterface(part1,part2,part3,null,lab);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
     * model for) the interface created by buildInterface().  This object will be initialized to a type
     * dependent value which is typically the the lowest valid non-negative value.
     **/
    abstract public ParameterValue createValue();
    
       
    /**
     * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
     * model for) the interface created by buildInterface().  This object will be initialized to value
     * given.  This value must be of the same class as returned by this routine, or some other class
     * that special provisions have been made for (i.e., a number of subclasses know how to deal with
     * Double objects).
     **/
    abstract public ParameterValue createValue(Object fromVal);
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* static variables for easy access to known/standard subclasses */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Standard type for a double value staying in the range 0.0...1.0. */
    public static final ParameterTypeDescriptor ZERO_TO_ONE =
    new ParameterTypeDescriptor.ZeroToOneType();
    
    /** Standard type for an unrestricted double value */
    public static final ParameterTypeDescriptor DOUBLE =
    new ParameterTypeDescriptor.DoubleType();
    
    /** Standard type for an unrestricted integer value. */
    public static final ParameterTypeDescriptor INTEGER =
    new ParameterTypeDescriptor.IntegerType();
    
    /** Standard type for a percentage (0.0..100.0) value. */
    public static final ParameterTypeDescriptor PERCENTAGE =
    new ParameterTypeDescriptor.PercentageType();
    
    /** Standard type for a Boolean value. */
    public static final ParameterTypeDescriptor BOOLEAN =
    new ParameterTypeDescriptor.BooleanType();
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* constructors */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Construct a descriptor for a kinetic effect parameter type.
     */
    public ParameterTypeDescriptor(String nm) {setName(nm);}
    
    /**
     * Construct a default descriptor for a kinetic typography parameter type.
     */
    public ParameterTypeDescriptor() {}
    
    /*-------------------------------------------------------------------*/
    /* Known / Standard subclasses */
    /*-------------------------------------------------------------------*/
    
    /**
     * Type descriptor for a double value staying between 0 and 1.
     */
    public static class ZeroToOneType extends DoubleRangeType  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        protected ZeroToOneType(String nm) {super(nm,0.0,1.0);}
        protected ZeroToOneType() {super("Value Between Zero and One", 0.0,1.0);}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    /*-------------------------------------------------------------------*/
    
    /**
     * Type descriptor for an arbitrary double value.
     */
    public static class DoubleType extends ParameterTypeDescriptor  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        protected DoubleType(String nm) {super(nm);}
        protected DoubleType() {super("Floating Point Value");}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create and return an interface for editing a value of this type.  The given object serves as the
         * model holding the value being edited and must be a DoubleHolder. The string
         * parameter here provides a human readable indication of what value is being edited.
         */
        public JComponent buildInterface(Object model, String lab) {
            // rectify the parameters
            if (lab == null) lab = " ";
            if (model == null) model = new DoubleHolder();
            
            // interface is a spinner
            JSpinner innerResult = new JSpinner(
            new SpinnerNumberModel(new Double(((DoubleHolder)model).value), null, null, new Double(1)));
            
            // attach the model listener
            innerResult.addChangeListener(new ParameterChangeListener(model,1.0));
            
            // package it up and ship it back
            return packageInterface(innerResult,lab);
        }
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
         * model for) the interface created by buildInterface().  In this case that is a DoubleHolder.
         */
        public ParameterValue createValue() {
            return new DoubleHolder();
        }
        
        /** 
         * Create and initialize an object that can hold a value of this type and that can be edited by
         * i.e., serve as the model for) the interface created by buildInterface().  In this case that 
         * is a DoubleHolder.  We can also accept Double and Integer values.
         */
        public ParameterValue createValue(Object fromVal) {
            if (fromVal != null)
            {
                if (fromVal instanceof DoubleHolder)
                    return new DoubleHolder(((DoubleHolder)fromVal).value);
                else if (fromVal instanceof Double)
                    return new DoubleHolder(((Double)fromVal).doubleValue());
                else if (fromVal instanceof Integer)
                    return new DoubleHolder(((Integer)fromVal).doubleValue());
            }
            // didn't get a value we can deal with go with a default value
            return createValue();
        }
    }
    
    /*-------------------------------------------------------------------*/
    /**
     * Type descriptor for an Double value confined to a range.  The default range
     * is 0.0..1.0 (so this is a lot like a ZeroToOne).
     */
    public static class DoubleRangeType extends ParameterTypeDescriptor  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* instance variables */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /** The minimum value that an instance of this type can have. */
        protected double minValue = 0.0;
        
        /** Get the minimum value that an instance of this type can have. */
        public double getMinValue() {return minValue;}
        
        /** Set the minimum value that an instance of this type can have. */
        public void setMinValue(double minV) {minValue = minV; /*xx update UI here */}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /** The maximum value that an instance of this type can have. */
        protected double maxValue = 1.0;
        
        /** Get the maximum value that an instance of this type can have. */
        public double getMaxValue() {return maxValue;}
        
        /** Set the maximum value that an instance of this type can have. */
        public void setMaxValue(double maxV) {maxValue = maxV; /*xx update UI here */}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        // public final double DOUBLE_SLIDER_SCALE = 100000.0;  //xx should be obsolete
        
        /**
         * Create and return an interface for editing a value of this type.  The given object serves as the
         * model holding the value being edited and must be a DoubleHolder.  The string
         * parameter here provides a human readable indication of what value is being edited.  Warning:
         * Once an interface has been built with this method it does not get updated if the type is changed
         * (e.g., if the min or max value is modified).
         */
        public JComponent buildInterface(Object model, String lab) {            
            // Create a slider
            ParameterSlider result = 
                new ParameterSlider(getMinValue(), getMaxValue(), ((DoubleHolder)model).value);
            
            // put in the listener to update the model
            result.addChangeListener(new ParameterChangeListener(model, 1.0));
            
            return packageInterface(result,lab);
        }
        
        /**
         * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
         * model for) the interface created by buildInterface().  In this case that is a DoubleHolder
         * initialized to the minimum value for the range.
         */
        public ParameterValue createValue() {
            return new DoubleHolder(getMinValue());
        }
        
        /** 
         * Create and initialize an object that can hold a value of this type and that can be edited by
         * i.e., serve as the model for) the interface created by buildInterface().  In this case that 
         * is a DoubleHolder.  We can also accept Double and Integer values.
         */
        public ParameterValue createValue(Object fromVal) {
            if (fromVal != null)
            {
                if (fromVal instanceof DoubleHolder)
                    return new DoubleHolder(((DoubleHolder)fromVal).value);
                else if (fromVal instanceof Double)
                    return new DoubleHolder(((Double)fromVal).doubleValue());
                else if (fromVal instanceof Integer)
                    return new DoubleHolder(((Integer)fromVal).doubleValue());
            }
            // didn't get a value we can deal with go with a default value
            return createValue();
         } 
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        public DoubleRangeType(String nm, double mn, double mx) {super(nm); setMinValue(mn); setMaxValue(mx);}
        public DoubleRangeType(double mn, double mx) {
            super("Floating point value in range"); setMinValue(mn); setMaxValue(mx);}
        public DoubleRangeType(String nm) {super(nm);}
        public DoubleRangeType() {super("Floating point value in range");}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    
    /*-------------------------------------------------------------------*/
    
    /**
     * Type descriptor for an integer value.
     */
    public static class IntegerType extends ParameterTypeDescriptor  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        protected IntegerType(String nm) {super(nm);}
        protected IntegerType() {super("Integer Value");}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create and return an interface for editing a value of this type.  The given object serves as the
         * model holding the value being edited and must be a IntHolder.  The string
         * parameter here provides a human readable indication of what value is being edited.
         */
        public JComponent buildInterface(Object model, String lab) {
            // rectify the parameters
            if (lab == null) lab = " ";
            if (model == null) model = new IntHolder();
            
            // interface is a spinner
            JSpinner innerResult = new JSpinner(
            new SpinnerNumberModel(new Double(((IntHolder)model).value), null, null, new Integer(1)));
                       
            // attach the model listener
            innerResult.addChangeListener(new ParameterChangeListener(model,1.0));

            // package it up and ship it back
            return packageInterface(innerResult,lab);
        }
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
         * model for) the interface created by buildInterface().  In this case that is an IntHolder.
         */
        public ParameterValue createValue() {
            return new IntHolder();
        }
                
        /** 
         * Create and initialize an object that can hold a value of this type and that can be edited by
         * i.e., serve as the model for) the interface created by buildInterface().  In this case that 
         * is a IntHolder.  We can also accept Integer values.
         */
        public ParameterValue createValue(Object fromVal) {
            if (fromVal != null)
            {
                if (fromVal instanceof IntHolder)
                    return new IntHolder(((IntHolder)fromVal).value);
                else if (fromVal instanceof Integer)
                    return new IntHolder(((Integer)fromVal).intValue());
            }
            // didn't get a value we can deal with go with a default value
            return createValue();
         } 
       
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    
    /*-------------------------------------------------------------------*/
    /**
     * Type descriptor for an interger value confined to a range.  The default range
     * is 0..99.
     */
    public static class IntegerRangeType extends ParameterTypeDescriptor  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* instance variables */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /** The minimum value that an instance of this type can have. */
        protected int minValue = 0;
        
        /** Get the minimum value that an instance of this type can have. */
        public int getMinValue() {return minValue;}
        
        /** Set the minimum value that an instance of this type can have. */
        public void setMinValue(int minV) {minValue = minV; /*xx update UI here */}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /** The maximum value that an instance of this type can have. */
        protected int maxValue = 99;
        
        /** Get the maximum value that an instance of this type can have. */
        public int getMaxValue() {return maxValue;}
        
        /** Set the maximum value that an instance of this type can have. */
        public void setMaxValue(int maxV) {maxValue = maxV; /*xx update UI here */}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create and return an interface for editing a value of this type.  The given object serves as the
         * model holding the value being edited and must be a IntHolder.  The string
         * parameter here provides a human readable indication of what value is being edited.  Warning:
         * Once an interface has been built with this method it does not get updated if the type is changed
         * (e.g., if the min or max value is modified).
         */
        public JComponent buildInterface(Object model, String lab) {
            
            // Create a slider
            ParameterSlider result = 
                new ParameterSlider(getMinValue(), getMaxValue(), ((IntHolder)model).value);
            
            // put in the listener to update the model
            result.addChangeListener(new ParameterChangeListener(model, 1.0));
            
            return packageInterface(result,lab);
        }
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
         * model for) the interface created by buildInterface().  In this case that is an IntHolder
         * initialized to the minimum for the range
         */
        public ParameterValue createValue() {
            return new IntHolder(getMinValue());
        }
                     
        /** 
         * Create and initialize an object that can hold a value of this type and that can be edited by
         * i.e., serve as the model for) the interface created by buildInterface().  In this case that 
         * is a IntHolder.  We can also accept Integer values.
         */
        public ParameterValue createValue(Object fromVal) {
            if (fromVal != null)
            {
                if (fromVal instanceof IntHolder)
                    return new IntHolder(((IntHolder)fromVal).value);
                else if (fromVal instanceof Integer)
                    return new IntHolder(((Integer)fromVal).intValue());
            }
            // didn't get a value we can deal with go with a default value
            return createValue();
         } 
       
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        public IntegerRangeType(String nm, int mn, int mx) {super(nm); setMinValue(mn); setMaxValue(mx);}
        public IntegerRangeType(int mn, int mx) {
            super("Integer value in range"); setMinValue(mn); setMaxValue(mx);}
        public IntegerRangeType(String nm) {super(nm);}
        public IntegerRangeType() {super("Integer value in range");}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    
    /*-------------------------------------------------------------------*/
    
    /**
     * Type descriptor for a percentage value.  This is basically a DoubleRange(0.0,100.0) with a slightly
     * different UI.
     */
    public static class PercentageType extends DoubleRangeType  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        protected PercentageType(String nm) {super(nm,0.0,100.0);}
        protected PercentageType() {super("Percentage",0.0,100.0);}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    
    /*-------------------------------------------------------------------*/
    
    /**
     * Type descriptor for a boolean value.
     */
    public static class BooleanType extends ParameterTypeDescriptor  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        protected BooleanType(String nm) {super(nm);}
        protected BooleanType() {}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create and return an interface for editing a value of this type.  The given object serves as the
         * model holding the value being edited and must be a BooleanHolder.  The string
         * parameter here provides a human readable indication of what value is being edited.
         */
        public JComponent buildInterface(Object model, String lab) {
            // fix missing parameters if needed
            if (lab == null) lab = " ";
            if (model == null) model = new BooleanHolder();
            
            // Make a labelled checkbox and put an unlabelled border on it
            JCheckBox innerResult = new JCheckBox(lab, ((BooleanHolder)model).value);
            
            // attach a listener to update the model
            innerResult.addChangeListener(new ParameterChangeListener(model, 1.0));
            
            return packageInterface(innerResult,null);
        }
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
         * model for) the interface created by buildInterface().  In this case that is a BooleanHolder.
         */
        public ParameterValue createValue() {
            return new BooleanHolder();
        }
                   
        /** 
         * Create and initialize an object that can hold a value of this type and that can be edited by
         * i.e., serve as the model for) the interface created by buildInterface().  In this case that 
         * is a BooleanHolder.  We can also accept Boolean values.
         */
        public ParameterValue createValue(Object fromVal) {
            if (fromVal != null)
            {
                if (fromVal instanceof BooleanHolder)
                    return new BooleanHolder(((BooleanHolder)fromVal).value);
                else if (fromVal instanceof Boolean)
                    return new BooleanHolder(((Boolean)fromVal).booleanValue());
            }
            // didn't get a value we can deal with go with a default value
            return createValue();
         } 
       
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    
    /*-------------------------------------------------------------------*/
    
    /**
     * Type descriptor for an enumerated value.  Values are represented by an ordered set of
     * names given as strings.
     */
    public static class EnumerationType extends ParameterTypeDescriptor  {
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* instance variables */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /** The list of strings providing the names of values for this type (in order) */
        protected String[] nameList = new String[0];
        
        /** Get the list of strings providing the names of values for this type. */
        public String[] getNameList() {return nameList;}
        
        /** Set the list of strings providing names of values for this type. */
        public void setNameList(String[] nmList) {
            if (nmList == null) nmList = new String[0];
            nameList = nmList;
        }
        
        /**
         * Set a particular value name.  Note the index of this name must be within the bounds
         * of the existing set of names.  If you need to change the number of values in this type
         * use setNameList instead.
         */
        public void setName(int indx, String newName) {nameList[indx] = newName;}
        
        /** Get the number of value in this enumerated type */
        public int numValues() {return nameList.length;}
        
        /**
         * Get the name for the value at the given ordinal position. Requests with out of bound values
         * will throw an error.
         */
        public String nameForValue(int val) {return nameList[val];}
        
        /**
         * Get the ordinal position corresponding to the given value name.  -1 is returned if
         * the given string is not a value name for this type.  Note: this currently
         * uses a simple linear search.  If you need good performance on big enumerations, you should
         * consider creating a subclass which uses a hash table.
         */
        public int valueForName(String nam) {
            if (nam != null) {
                for (int i = 0; i < getNameList().length; i++) {
                    if (nam.equals(nameForValue(i))) return i;
                }
            }
            return -1;
        }
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create and return an interface for editing a value of this type.  The given object serves as the
         * model holding the value being edited and must be a IntHolder.  The string
         * parameter here provides a human readable indication of what value is being edited.  Warning:
         * Once an interface has been built with this method it does not get updated if the type is changed
         * (e.g., if the enumeration name list is changed).
         */
        public JComponent buildInterface(Object model, String lab) {
            // make a combo box for the values
            JComboBox innerResult = new JComboBox(getNameList());
            innerResult.setBackground(Color.white);
            
            // attach a listener to update its value
            innerResult.addActionListener(new ParameterChangeListener(model,1.0));
            
            return packageInterface(innerResult,lab);
        }
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        /**
         * Create an object that can hold a value of this type and that can be edited by (i.e., serve as the
         * model for) the interface created by buildInterface().  In this case that is an IntHolder
         * initialized to zero (which corresponds to the first enumerated value).
         */
        public ParameterValue createValue() {
            return new IntHolder(0);
        }
        
                             
        /** 
         * Create and initialize an object that can hold a value of this type and that can be edited by
         * i.e., serve as the model for) the interface created by buildInterface().  In this case that 
         * is a IntHolder.  We can also accept Integer values.
         */
        public ParameterValue createValue(Object fromVal) {
            if (fromVal != null)
            {
                if (fromVal instanceof IntHolder)
                    return new IntHolder(((IntHolder)fromVal).value);
                else if (fromVal instanceof Integer)
                    return new IntHolder(((Integer)fromVal).intValue());
            }
            // didn't get a value we can deal with go with a default value
            return createValue();
         } 
       
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        /* constructors */
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
        
        public EnumerationType(String nm, String[] namList) {super(nm); setNameList(namList);}
        public EnumerationType(String[] namList) {super("Choose From"); setNameList(namList);}
        public EnumerationType(String nm) {super(nm);}
        public EnumerationType() {super("Choose From");}
        
        /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    }
    
    /*-------------------------------------------------------------------*/
    
    /**
     * Object that listens for changes to one of the components used to control
     * a parameter and updates a model object to reflect a new value.  This 
     * class is tuned to the specifics of the components used in the interfaces 
     * for editing effect parameter values (i.e., those created in the buildInterface()
     * methods of sibling *Type classes).  If the details of those interfaces change, this 
     * class may need to change also.  
     */
    protected class ParameterChangeListener implements ChangeListener, ActionListener {
        /** Should we be displaying a double value (vs. an int) */
        protected boolean isDouble = false;
        
        /** 
         * Scale factor that our slider is using (values stored in slider are divided by this 
         * to get the display value).
         */
        protected double scaleFactor = 1.0;
        
        /** Model object that we are updating */ 
        protected Object model = null;
        
        public ParameterChangeListener (Object mod, double sclFactor) {
            scaleFactor = sclFactor;
            model = mod;
            if (model instanceof DoubleHolder)
            {
                isDouble = true;
            }
            else if (model instanceof IntHolder || model instanceof BooleanHolder)
            {
                isDouble = false;
            }
            else
            {
                isDouble = false;
                // this will throw an exception 
                ((DoubleHolder)mod).value = 0.0;
            }
        }
        
        public void stateChanged(ChangeEvent changeEvent) 
        {
            Object sender = changeEvent.getSource();
            
            //xx note: probably need to handle undo/redo in here
            
            // look at the type of the sender to figure out what to do
            if (sender instanceof JSpinner)
            {
                // extract the spinner and get its current value
                JSpinner spin = (JSpinner)sender;
                if (isDouble)
                    ((DoubleHolder)model).value = ((Double)spin.getValue()).doubleValue();
                else
                    ((IntHolder)model).value = ((Integer)spin.getValue()).intValue();
            }
            else if (sender instanceof JCheckBox)
            {
                // extract the checkbox and get its current value
                JCheckBox box = (JCheckBox)sender;
                ((BooleanHolder)model).value = box.isSelected();
            }
            else if (sender instanceof ParameterSlider)
            {
                // extract the slider and get its current value
                ParameterSlider slide = (ParameterSlider)sender;
                
                if (isDouble)
                    ((DoubleHolder)model).value = slide.getDoubleValue();
                else
                    ((IntHolder)model).value = (int)slide.getValue();
            } 
            
            // tell the main editor that an edit has occured so we modify the preview
            KTEdit.getInstance().editOccured();
        } 
        
      public void actionPerformed(ActionEvent evt) 
      {
            Object sender = evt.getSource();
            
            //xx note: probably need to handle undo/redo in here
            
            // look at the type of the sender to figure out what to do
            if (sender instanceof JComboBox)
            {
                // extract the combobox and get its current value
                JComboBox box = (JComboBox)sender;
                
                ((IntHolder)model).value = box.getSelectedIndex();
            }
                        
            // tell the main editor that an edit has occured so we modify the preview
            KTEdit.getInstance().editOccured();
      }
 
    }
    
    /*-------------------------------------------------------------------*/
    
    /**
     * Simple class to hold a mutable double value to serve as a model for
     * interfaces created by type descriptors.
     */
    public static class DoubleHolder implements ParameterValue {
        public double value;
        public DoubleHolder() {}
        public DoubleHolder(double v) {value = v;}
        
        /** Get the value.  The actual return type here will always be a Double. */
        public Object getValue() {return new Double(value);}
        
        /** Set the value from a double. */
        public void setValue(double val) {value = val;}
        
        /**
         * Set the value from an Object.  The actual class of this object must
         * be a Double, Integer, or Boolean.  For Double,
         * Integer, or Boolean values, this is equivalent to calling
         * the similarly typed method.
         */
        public void setValue(Object val) {
            if (val instanceof Double) 
                setValue((Double)val);
            else if (val instanceof Integer)
                setValue((Integer)val);
            else if (val instanceof Boolean)
                setValue((Boolean)val);
            else
                // this will throw an exception
                setValue((Double)val);
        }
        
        /** 
         * Set the value from a Boolean. For false this sets the value to 0.0.
         * For true this sets the value to 1.0.
         */
        public void setValue(Boolean val) {setValue(val.booleanValue()?1.0:0.0);}
        
        /**
         * Set the value from a Double.
         */
        public void setValue(Double val) {setValue(val.doubleValue());}
        
        /** 
         * Set the value from a Boolean. For false this sets the value to 0.0.
         * For true this sets the value to 1.0.
         */
        public void setValue(boolean val) {setValue(val?1.0:0.0);}
        
        /** Set the value from an Integer. */
        public void setValue(Integer val) {setValue(val.doubleValue());}
        
        /** Set the value from an int. */
        public void setValue(int val) {setValue((double)val);}        
    }
    
    /**
     * Simple class to hold a mutable int value to serve as a model for
     * interfaces created by type descriptors.
     */
    public static class IntHolder implements ParameterValue {
        public int value;
        public IntHolder() {}
        public IntHolder(int v) {value = v;}
                
        /** Get the value.  The actual return type here will always be a Integer. */
        public Object getValue() {return new Integer(value);}
        
        /** Set the value from a double. */
        public void setValue(double val) {value = (int)val;}
        
        /**
         * Set the value from an Object.  The actual class of this object must
         * be a Double, Integer, or Boolean.  For Double,
         * Integer, or Boolean values, this is equivalent to calling
         * the similarly typed method.
         */
        public void setValue(Object val) {
            if (val instanceof Double) 
                setValue((Double)val);
            else if (val instanceof Integer)
                setValue((Integer)val);
            else if (val instanceof Boolean)
                setValue((Boolean)val);
            else
                // this will throw an exception
                setValue((Integer)val);
        }
        
        /** 
         * Set the value from a Boolean. For false this sets the value to 0.
         * For true this sets the value to 1.
         */
        public void setValue(Boolean val) {setValue(val.booleanValue()?1:0);}
        
        /**
         * Set the value from a Double.
         */
        public void setValue(Double val) {setValue(val.intValue());}
        
        /** 
         * Set the value from a Boolean. For false this sets the value to 0.
         * For true this sets the value to 1.
         */
        public void setValue(boolean val) {setValue(val?1:0);}
        
        /** Set the value from an Integer. */
        public void setValue(Integer val) {setValue(val.intValue());}
        
        /** Set the value from an int. */
        public void setValue(int val) {value = val;}  
    }
    
    /**
     * Simple class to hold a mutable boolean value to serve as a model for
     * interfaces created by type descriptors.
     */
    public static class BooleanHolder implements ParameterValue {
        public boolean value;
        public BooleanHolder() {}
        public BooleanHolder(boolean v) {value = v;}
                        
        /** Get the value.  The actual return type here will always be a Boolean. */
        public Object getValue() {return new Boolean(value);}
        
        /** Set the value from a double. Values of 0.0 give false all other give true. */
        public void setValue(double val) {value = val != 0.0;}
        
        /**
         * Set the value from an Object.  The actual class of this object must
         * be a Double, Integer, or Boolean.  For Double,
         * Integer, or Boolean values, this is equivalent to calling
         * the similarly typed method.
         */
        public void setValue(Object val) {
            if (val instanceof Double) 
                setValue((Double)val);
            else if (val instanceof Integer)
                setValue((Integer)val);
            else if (val instanceof Boolean)
                setValue((Boolean)val);
            else
                // this will throw an exception
                setValue((Boolean)val);
        }
        
        /** 
         * Set the value from a Boolean. 
         */
        public void setValue(Boolean val) {setValue(val.booleanValue());}
        
        /**
         * Set the value from a Double. Values of 0.0 give false all other give true.
         */
        public void setValue(Double val) {setValue(val.doubleValue()!=0.0);}
        
        /** 
         * Set the value from a Boolean. For false this sets the value to 0.
         * For true this sets the value to 1.
         */
        public void setValue(boolean val) {value = val;}
        
        /** Set the value from an Integer. Non-zero is considered true. */
        public void setValue(Integer val) {setValue(val.intValue()!=0);}
        
        /** Set the value from an int. */
        public void setValue(int val) {setValue(val!=0);}  
    }
    
    /*-------------------------------------------------------------------*/
}
