package KTEditor;

import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import java.util.Enumeration;
 
/**
 * This action subclass toggles the style attribute which signifies a
 * particular effect across the current selection. If
 * all characters in the selection already have the attribute
 * for this effect set, then this removes the attribute from the selection.
 * We also remove the attributes corresponding to any conflicting effects.
 * Note: if the selection is empty these actions all apply to the
 * "input attributes" (which control text about to be entered).  <p>
 *
 * Effects differ from tags somewhat because effects include a set of
 * parameter values.  Hence there are some special rules about how effects
 * conflict and how effects are merged, so that we preserve the right set of
 * parameter values with the attribute.  To handle this, this class is set up
 * to take an EffectDescriptor as its attribute.  However, this
 * EffectDescriptor is always instantiated into an EffectInstanceDescriptor
 * (which manages a set of actual parameters), and that EffectInstanceDescriptor
 * is what is used to mark the text.  We do this by doing a pass over the
 * selection area first to determine whether we are creating a completely
 * new instance, or merging with (reusing) an instance that is already in
 * the document.  The current merge rule is that if the selection does not
 * intersect or directly abut the left edge of an existing effect instance,
 * then an entirely new effect instances is instantiated.  However, if the
 * selection intersects or abuts the left edge of one or more existing effects
 * then the instance corresponding to the first of these is used for this
 * operation -- basically we expand the left-most instance that we intersect
 * across the entire selection area.
 */
public class ToggleEffectAttribute extends ToggleTagAttribute {
    /**
     * The effect instance that we will be using for the current operation.
     * this gets set by xx at the start of an operation and is not good across
     * operations.
     */
    protected EffectInstanceDescriptor instance = null;
    
    /**
     * Get the descriptor for the effect instance that we will be toggling.
     * In this subclass we do not normally use the SegmentDescriptor
     * provided by setTag().  Instead, when that descriptor is an
     * EffectDescriptor we maintain an appropriate EffectInstanceDescriptor
     * and return that instead.
     */
    public SegmentDescriptor getTag() {
        // if we have a prepared instance, return it
        if (instance != null) return instance;
        
        // otherwise fall back on our normal tag
        return tag;
    }
    
    /**
     * Construct an action for toggling attributes marking text with
     * the given effect. */
    public ToggleEffectAttribute(SegmentDescriptor tg) { super(tg); }
    
    /**
     * Test whether the given attribute set contains an effect instance that the
     * effect associated with this object can be merged into.  If such an effect 
     * instance is found in the set, it is returned, otherwise null is returned.
     */
    public EffectInstanceDescriptor findMergeEffect(AttributeSet inAttrs)
    {        
        // look at each attribute set here and see if its an effect
        for (Enumeration en = inAttrs.getAttributeNames(); en.hasMoreElements(); ) 
        {  
            Object attr = en.nextElement();                  
            // is it an effect with the same name (and not set to false)                  
            if (attr instanceof EffectInstanceDescriptor &&                
                getTag().getName().equals(((EffectInstanceDescriptor)attr).getName()) &&
                inAttrs.getAttribute(attr) instanceof EffectInstanceDescriptor)                    
            {                     
                // we found one and we are done
                return  (EffectInstanceDescriptor)attr;     
            }           
        }
        
        // nothing found
        return null;
    }
    
    //xx reconsider the abuts left rule...
    
    /**
     * Actaully carry out the action. This first has a look at the area of
     * the selection to determine whether to merge with an existing effect
     * instance in the document, or instantiate an entirely new instance.
     * it then uses the instance as the attribute and carries out the toggle
     * action.<p>
     *
     * The merge rule for effects is currently that if the selection area
     * overlaps or abuts the left end of one or more effects with the same
     * name, then we merge with the first (left-most) one of those.  Otherwise,
     * we instantiate a completely new effect instance.<p>
     *
     * If all of the current selection is not already marked with our
     * attribute, then we mark it.  If all parts of it are already
     * marked then we unmark it.  Any attributes which conflict with
     * our attribute are removed.  In the case of the selection being
     * empty, these actions apply instead to the "input attributes" (i.e.,
     * the attributes that will be applied to text typed at the insertion
     * point).
     */
    public void actionPerformed(ActionEvent actEvt) {
        // Pull the editor, document, and editorkit and ensure they exist
        JEditorPane edPane = getEditor(actEvt);
        if (edPane == null) return;
        StyledEditorKit edKit = getStyledEditorKit(edPane);
        if (edKit == null) return;
        DefaultStyledDocument doc = (DefaultStyledDocument)edPane.getDocument();
        if (doc == null) return;
        
        // clear out any old effect instance
        instance = null;
        
        // get the selection
        int selStart = edPane.getSelectionStart();
        int selEnd = edPane.getSelectionEnd();
        
        // if the selection is empty then this is always a new instance
        if (selStart == selEnd) {
            instance = new EffectInstanceDescriptor((EffectDescriptor)getTag());
        }
        else 
        {
            int pos;               // position in text we are currently looking at
            Element elm;           // text chunk we are currently looking at
            
            // test for the "abuting the left edge rule"
            pos = selStart;
            if (selStart > 0) 
            {                
                // is there a merge at the start
                elm = doc.getCharacterElement(pos);
                instance = findMergeEffect(elm.getAttributes());
                
                // if no merge at start we test prior to the selection
                if (instance == null)
                {
                    instance = findMergeEffect(doc.getCharacterElement(pos-1).getAttributes());  
                    //xx internationalization direction?
                }
                
                // move past first chunk
                pos = elm.getEndOffset();
            }
            
            // walk down the rest of the selection and see if we overlap 
            // anybody we could merge with.
            for (; (pos < selEnd) && (instance == null) ; pos = elm.getEndOffset()) 
            { 
                // pull out the element 
                elm = doc.getCharacterElement(pos);
                
                // look for a merge with our effect in the current chunk
                instance = findMergeEffect(elm.getAttributes());
            }
            
            // if we make it out of our search without finding an instance
            // then we should create a fresh one
            if (instance == null)
                instance = new EffectInstanceDescriptor((EffectDescriptor)getTag());
        }
        
        // we have now established the right effect instance to work with
        // let the superclass do the rest of the work
        super.actionPerformed(actEvt);
        
        // drop the current instance so its not hanging around between actions
        instance = null;
        
        // tweek the selection so we get an update to the listener and put up 
        // the new effect interface
        edPane.setSelectionStart(edPane.getSelectionStart());
    }
}

